/* 

   Xwindows interface for NLSL program  
   Written by Diane Crepeau and David Budil 2/8/93

This file contains the following routines:

 *****************************
 * Fortran-callable routines *
 *****************************

void initwindows()
  Initializes NLS interface to XWindows

int newwindow( banner titles[] )
  Sets up a new NLS window (returns number of active windows)

int killwindow()
  Removes last NLS window (returns number of active windows)

void wpoll()
  Poll all active windows for events

void shutwindows()
  Close all open windows and disconnect from display

void pltwin( exptl, resid, cmpts, sfac, ldcmpt, npt, nsi, wnum )
  Plot relevant data in a selected window

 ************************
 * Internal subroutines *
 ************************
void getNLSLcolor( struct nlswindow win, char *colorstr, XColor *color )
               Returns a color that is requested by name 

void yminmax( double array[], int n, double *min, double *max )
               Finds the minimum and maximum in a data array

void ScaleNLSLplot( struct nlswindow win )
               Finds suitable scale factors for selected window

void DrawSpectra( struct nlswindow win )
               Plots the spectra in a given window

void NLSLevent( struct nlswindow win )
               Handle various events for the given NLSL window

   KEYSTROKE OPERATIONS: 

     "a": Show "all": experimental, simulated, and difference spectrum
     "b": Show "both" experimental and simulated spectrum (default)
     "c": Show component spectra only
     "d": Show the difference spectrum only
     "e": Show the experimental spectrum only
     "s": Show the simulation only

      (may want to revise: a,d,s,c,z toggle axes, data, simulation, components,
       and zero-line)

   MOUSE-BUTTON PRESSES :(not yet implemented in pltd.c)

   Right button:  When pressed, the rightheand mouse button will highlight
        a pixel point and display the data coordinates (x, y).

   Left button:  When pressed, moved, and released, the lefthand mouse 
        button will highlight the pixel points at the spots where the 
        button was pressed and released, and will display the difference 
        between the two points (x, y).

   The function will also automatically rescale the data if the
   window size is changed.

   Written by Diane Crepeau 2/8/93            
   MODIFIED by David Budil 

*************************************************************************/


#define XCALLS 1

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdio.h>
#include <string.h>

#define MAX_WINDOWS 5       /* Maximum number of windows allowed */
#define MAX_SITES 3         /* Maximum number of components in spectrum */
#define EDGE 0.1            /* border around axes (frac. of window dim.) */
 
#define DATA 1              /* Bit masks for "show" flag */
#define SIMULATION 2
#define COMPONENTS 4
#define DIFFERENCE 8
#define AXES 16
                          /* Window dimensions--all in units of pixels */
#define NLSL_border 5     /* width of window border */
#define STD_X_LOC 50      /* standard x-position for new window (upper left) */
#define STD_Y_LOC 300     /* standard y-position for new window (upper left) */
#define STD_WIDTH 400     /* standard width for a window */
#define STD_HEIGHT 267    /* standard height for a window */

typedef char banner[40];

/* NLSL window structure definitions */

struct nlswindow
{
  Display *dspy;
  Drawable wdw;
  GC gc;                       /* Graphics context */
  Colormap map;                /* Window colormap */
  int screen;                  /* Screen number */ 
  int nx,ny;                   /* Dimensions of window in pixels */ 
  int led,red,bed,ted,y0;      /* Pixel coords of inner edges, zero line */
  float xscale,yscale;      /* Plot scaling information */
  double *expt, *diff, *scale;
  double *comp[MAX_SITES];
  XPoint *pxexpt, *pxsim, *pxdiff;
  XPoint *pxcomp[MAX_SITES];
  int npts, ncmp;              /* Number of expt points in arrays */
  unsigned show;               /* Flag indicating screen contents */
  unsigned fgnd,bgnd;          /* Set to default foreground/bckground colors */
};



/* Variable declarations */

struct nlswindow wds[MAX_WINDOWS];
XSizeHints sizehints[MAX_WINDOWS];
XColor orange, green, magenta, black, white;
int n_win = 0;
int DisplayOK = 0;
int opened[MAX_WINDOWS];

/*******************************************************
*                                                      * 
*   getNLSLcolor: Fetch a color using X-Windows parser *
*                                                      * 
********************************************************/  

#ifdef XCALLS
void getNLSLcolor( struct nlswindow win, char *colorstr, XColor *color )
{
Status parsed,allocd;
   parsed = XParseColor( win.dspy, win.map, colorstr, color );
   if (parsed) allocd = XAllocColor( win.dspy, win.map, color );
   if ( !(parsed && allocd) ) *(color) = white;

}
#endif


void initwindows(void )
{
int wn = 0;
   DisplayOK = getenv( "DISPLAY" );
   if ( !DisplayOK ) return;


     /* initialize size hints */

   for (wn=0; wn<MAX_WINDOWS; wn++) {

     /* Initialize size hints for window layout */

      sizehints[wn].width = STD_WIDTH;
      sizehints[wn].height = STD_HEIGHT;
      sizehints[wn].x = STD_X_LOC + wn*STD_HEIGHT/5;
      sizehints[wn].y = STD_Y_LOC + wn*STD_WIDTH/5;
      sizehints[wn].flags = PPosition | PSize;

     /* Set display, screen, and colormap for the new window */

#ifdef XCALLS
      wds[wn].dspy   = XOpenDisplay( "" );
      wds[wn].screen = DefaultScreen( wds[wn].dspy );
      wds[wn].map    = XDefaultColormap( wds[wn].dspy, wds[wn].screen );
#endif

    }
    n_win=0;
}




/***************************************************
*                                                  * 
*   newwindow: Create a window for NLS display     *
*                                                  * 
***************************************************/  

#ifdef XCALLS

int newwindow( banner title )
{
int i;

   if (!DisplayOK) return;

   /* Create window with properties set in sizehints  */

   wds[n_win].wdw = XCreateSimpleWindow(
                  wds[n_win].dspy,
                  DefaultRootWindow( wds[n_win].dspy ),
                  sizehints[n_win].x, sizehints[n_win].y,
                  sizehints[n_win].width, sizehints[n_win].height, NLSL_border,
                  wds[n_win].fgnd, wds[n_win].bgnd );

   /*
   This interface to the window manager is used in 
   Version 11 Release 4 and earlier 
   */

   XSetStandardProperties( wds[n_win].dspy, wds[n_win].wdw, title, title,
                           None, "/", 1, sizehints[n_win] );

   /* Find color values */

   getNLSLcolor( wds[n_win], "orange",  &orange );
   getNLSLcolor( wds[n_win], "green",   &green  );
   getNLSLcolor( wds[n_win], "magenta", &magenta );
   getNLSLcolor( wds[n_win], "white",   &white );
   getNLSLcolor( wds[n_win], "black",   &black );
      
   /* Create Graphics Context for the window */

   wds[n_win].gc = XCreateGC(wds[n_win].dspy, wds[n_win].wdw, 0, 0);
   XSetForeground( wds[n_win].dspy, wds[n_win].gc, white );

   /* Select input events: keypresses, mouse button presses, exposures */

   XSelectInput( wds[n_win].dspy, wds[n_win].wdw,
                 ButtonPressMask | KeyPressMask | ExposureMask );

   /* Map window "on top of" existing windows */

   XMapRaised( wds[n_win].dspy, wds[n_win].wdw ); 

   /* Initialize flags and data array pointers */

   wds[n_win].show = DATA | SIMULATION | AXES; 
   wds[n_win].expt = wds[n_win].diff = wds[n_win].scale = NULL;
   for (i=0; i<MAX_SITES; i++) wds[n_win].comp[i] = NULL;
   n_win++;

   return n_win;
} 

#else

/* Alternative "dummy" version when XLIB routines are not to be called
   (used mainly for debugging) */

int newwindow(  banner title )
{
int i;

   /* Initialize flags and data array pointers */

   if (n_win < MAX_WINDOWS) {
      wds[n_win].show = DATA | SIMULATION;
      wds[n_win].nx = STD_WIDTH;
      wds[n_win].ny = STD_HEIGHT;
      wds[n_win].expt = wds[n_win].diff = wds[n_win].scale = NULL;
      for (i=0; i<MAX_SITES; i++) wds[n_win].comp[i] = NULL;
      n_win++;
      }
   return n_win;
}
#endif



/******************************************
*                                         * 
*  killwindow Remove last NLSL window     *
*                                         * 
******************************************/

#ifdef XCALLS

int killwindow( void )
{
XWindowAttributes attribs;

  if ( n_win > 0) {
	n_win--;

        /* Save current window size/location in hints */

        XGetWindowAttributes( wds[n_win].dspy, wds[n_win].wdw, &attribs ); 
        sizehints[n_win].height = attribs.height;
        sizehints[n_win].width = attribs.width;
        sizehints[n_win].x = attribs.x;  
        sizehints[n_win].y = attribs.y;

        XFreeGC( wds[n_win].dspy, wds[n_win].gc );
	XDestroyWindow( wds[n_win].dspy, wds[n_win].wdw );
	XCloseDisplay( wds[n_win].dspy );
      } 
  return n_win;
}
#endif


/*****************************************************************
 *                                                               *
 * yminmax: Find minimum and maximum in a given double-precision *
 *          array                                                *
 *                                                               *
 *****************************************************************/

void yminmax( double array[], int n, double *min, double *max )
{
int i;
   for (i=0; i<n; i++) {
      *max = ( array[i] > *max ) ? array[i] : *max;
      *min = ( array[i] < *min ) ? array[i] : *min;
      }
}

/*********************************************************************
 *                                                                   *
 *  scaleNLSLplot: Determine scale factors from data to pixel units  *
 *                                                                   *
 *********************************************************************/

void ScaleNLSLplot( struct nlswindow *win )
{
int i;
double val, ymin, ymax;
XWindowAttributes attribs;

#ifdef XCALLS
   XGetWindowAttributes( win->dspy, win->wdw, &attribs ); 
   win->ny = attribs.height;          /* Set ny to window height in pixels */
   win->nx = attribs.width;           /* Set nx to window width in pixels */
#else
   win->nx = STD_WIDTH;
   win->ny = STD_HEIGHT;
#endif

   /* Set edges ( in pixel units ) */

   win->led = win->nx * EDGE;
   win->red = win->nx * ( 1.0 - EDGE );
   win->ted = win->ny * EDGE;
   win->bed = win->ny * ( 1.0 - EDGE );

  
   /* Find minimum maximum y-value in arrays that will be plotted */

   ymin = ymax = 0.0;
   if (win->show & DATA)       yminmax( win->expt, win->npts, &ymin, &ymax );
   if (win->show & DIFFERENCE) yminmax( win->diff, win->npts, &ymin, &ymax );
      
   if (win->show & COMPONENTS && !(win->show & SIMULATION) ) 
      for (i=0; i<win->ncmp; i++) 
	 yminmax( win->comp[i], win->npts, &ymin, &ymax );

   if (win->show & SIMULATION) {
      for (i=0; i<win->npts; i++) {
         val = win->expt[i] - win->diff[i];
         ymax = ( val > ymax ) ? val : ymax;
         ymin = ( val < ymin ) ? val : ymin;
         } 
      }

   win->yscale = (win->bed - win->ted) / (ymax-ymin);
   win->xscale = (win->red - win->led) / (double) (win->npts-1);
   win->y0     = (int) win->bed + win->yscale * ymin;
}


/******************************************************************
 *                                                                * 
 *  DrawSpectra: Display appropriate spectra in the given window  *
 *                                                                * 
 *****************************************************************/

void DrawSpectra( struct nlswindow *win )
{
int i, ic;

   if (!DisplayOK) return;

#ifdef XCALLS
   XClearWindow( win->dspy, win->wdw );
#endif

   ScaleNLSLplot( win );

   /* Draw axes if "show" flag is set */
  
   if (win->show & AXES)  {

#ifdef XCALLS
      XSetForeground( win->dspy, win->gc, white );
      XDrawLine( win->dspy, win->wdw, win->gc, 
                 win->led, win->ted, win->red, win->ted );
      XDrawLine( win->dspy, win->wdw, win->gc, 
                 win->led, win->bed, win->red, win->bed );
      XDrawLine( win->dspy, win->wdw, win->gc, 
                 win->led, win->ted, win->led, win->bed );
      XDrawLine( win->dspy, win->wdw, win->gc, 
                 win->red, win->ted, win->red, win->bed );
      XDrawLine( win->dspy, win->wdw, win->gc, 
                 win->led, win->y0, win->red, win->y0 );
#endif
   }

/* Calculate and draw pixels for data if "show" flag requires it */ 

      if (win->show & DATA) {
         if (win->pxexpt == NULL ) 
            win->pxexpt = (XPoint *) calloc( win->npts, sizeof( XPoint ) );
         for (i=0; i<win->npts; i++) {
            win->pxexpt[i].x = win->led + win->xscale * i;
            win->pxexpt[i].y = win->y0 - (win->expt[i] * win->yscale);
         }

#ifdef XCALLS
      XSetForeground( win->dspy, win->gc, orange );
      XDrawLines( win->dspy, win->wdw, win->gc, win->pxexpt, win->npts,
	          CoordModeOrigin );
#endif
    }

/* Calculate and draw pixels for simulation if "show" flag requires it */ 

  if (win->show & SIMULATION) {
      if (win->pxsim == NULL ) 
         win->pxsim = (XPoint *) calloc( win->npts, sizeof( XPoint ) );
      for (i = 0; i < win->npts; i++) {
         win->pxsim[i].x = win->led +  win->xscale * i;
         win->pxsim[i].y = win->y0 - ( win->expt[i] - win->diff[i] )
                                     * win->yscale;
         }
#ifdef XCALLS
      XSetForeground( win->dspy, win->gc, green );
      XDrawLines( win->dspy, win->wdw, win->gc, win->pxsim, win->npts,
	          CoordModeOrigin );
#endif
  }


/* Calculate and draw pixels for difference spectrum if needed */ 

  if (win->show & DIFFERENCE) {
     if (win->pxdiff == NULL )
         win->pxdiff = (XPoint *) calloc( win->npts, sizeof( XPoint ) );
     for (i=0; i<win->npts; i++) {
        win->pxdiff[i].x = win->led + win->xscale*i;
        win->pxdiff[i].y = win->y0 - (win->diff[i] * win->yscale);
     }
#ifdef XCALLS
     XSetForeground( win->dspy, win->gc, green );
     XDrawLines( win->dspy, win->wdw, win->gc, win->pxdiff, win->npts,
	          CoordModeOrigin );
#endif
  }

/* Calculate and draw pixels for component specta if needed */
  
  if ( (win->show & COMPONENTS) && (win->ncmp>1) )  {
      for (ic=0; ic < win->ncmp; ic++) {
         if ( win->pxcomp[ic] == NULL )
            win->pxcomp[ic] = (XPoint *) calloc( win->npts,sizeof( XPoint ) );
         for (i=0; i<win->npts; i++) {
            win->pxcomp[ic][i].x = win->led + win->xscale*i;
            win->pxcomp[ic][i].y = win->y0 - (  win->scale[i] * win->yscale
                                               * win->comp[ic][i] );
#ifdef XCALLS
	    XSetForeground( win->dspy, win->gc, green );
	    XDrawLines( win->dspy, win->wdw, win->gc, win->pxcomp[ic],
                        win->npts, CoordModeOrigin );
#endif
         }
       }
    }

#ifdef XCALLS
    XFlush( win->dspy );
#endif
}

/*******************************************
*                                          *
* NLSLevent: Event handler for NLS windows *
*                                          *
********************************************/

void NLSLevent( struct nlswindow *win )
{

#ifdef XCALLS
   XEvent myevent;
   KeySym mykey;              
   XWindowAttributes attribs;
   char txt[10];
   int i;

 /* read the next event */

   XNextEvent( win->dspy, &myevent );

  switch (myevent.type) {

    /* Repaint window on expose events */

    case Expose:
       if (myevent.xexpose.count == 0) {
          XGetWindowAttributes( win->dspy, win->wdw, &attribs );
          win->ny = attribs.height;
          win->nx = attribs.width;
          DrawSpectra( win );
       } 
       break;

    /* Process keyboard mapping changes */

    case MappingNotify:
       XRefreshKeyboardMapping( &myevent );
       break;

    /* Process mouse-button presses */

/* DC's stuff  *********************************************************


    case ButtonPress:          
      drawln(wn); 
      if (myevent.xbutton.button == 1) {
        xpt = myevent.xbutton.x;
        ypt = myevent.xbutton.y;

        calcpt(xpt, ypt, &xpt1, &ypt1, wn);
        setxpts(pts1, xpt, ypt);

        XDrawPoints (wds[wn].dspy, wds[wn].wndw, wds[wn].gc, pts1, 5, 
                     wds[wn].mode);
        }
      if (myevent.xbutton.button == 3)
      {
        xpt = myevent.xbutton.x;
        ypt = myevent.xbutton.y;

        calcpt(xpt, ypt, &xpt1, &ypt1, wn);
        setxpts(pts1, xpt, ypt);

        getstr(cpltstr, xpt1, ypt1);

        XDrawPoints (wds[wn].dspy, wds[wn].wndw, wds[wn].gc, pts1, 5,
                     wds[wn].mode);
        if ((xpt + (strlen(cpltstr) * 6)) < wds[wn].nx)
//                                Doesn't overlap right edge
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xpt + 6,
                                 ypt + 3, cpltstr, strlen(cpltstr));
        else if ((xpt - (strlen(cpltstr) * 6)) > 0)
        {
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xpt - 
                                 (strlen(cpltstr) * 6) + 6, ypt + 3, cpltstr, 
                                 strlen(cpltstr));
	}
        else if ((ypt + 10) > wds[wn].ny)
        {
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xpt - 
                                 (strlen(cpltstr) * 6) / 2 + 6,
                                 ypt + 10, cpltstr, strlen(cpltstr));
	}
        else
        {
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xpt - 
                                 (strlen(cpltstr) * 6) / 2 + 6,
                                 ypt - 10, cpltstr, strlen(cpltstr));
	}
      }
      break; 

    case ButtonRelease:
      if (myevent.xbutton.button == 1)
      {
        xptr = myevent.xbutton.x;
        yptr = myevent.xbutton.y;

        xlow = (xpt - xptr) / 2 + xptr;
        ylow = (ypt - yptr) / 2 + yptr;

        calcpt(xptr, yptr, &xpt2, &ypt2, wn);
        setxpts(pts2, xptr, yptr);

        xptd = xpt1 - xpt2;
        yptd = ypt1 - ypt2;

        getstr(cpltstr, xptd, yptd);

        XDrawPoints (wds[wn].dspy, wds[wn].wndw, wds[wn].gc, pts2, 5,
                     wds[wn].mode);

        if ((xlow + 250) < wds[wn].nx)
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xlow + 6,
                                 ylow + 3, cpltstr, strlen(cpltstr));
        else if ((xlow - 250) > 0)
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xlow - 
                                 (strlen(cpltstr) * 6) + 6,
                                 ylow + 3, cpltstr, strlen(cpltstr)); 
        else if ((ylow + 10) > wds[wn].ny)
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xlow - 
                                 (strlen(cpltstr) * 6) / 2 + 6,
                                 ylow + 10, cpltstr, strlen(cpltstr));
        else
          XDrawString(wds[wn].dspy, wds[wn].wndw, wds[wn].gc, xlow - 
                                 (strlen(cpltstr) * 6) / 2 + 6,
                                 ylow - 10, cpltstr, strlen(cpltstr));
      }
      break;
  

**********************************************************************/

  
    /* Process the most recent key press */

    case KeyPress:
      i = XLookupString( &myevent, txt, 10, &mykey, 0 );
      if (i >= 1) {
          switch ( txt[i-1] ) {

	  case 'a':
	    win->show = DATA | SIMULATION | DIFFERENCE;
	    break; 

	  case 'b':
	    win->show = DATA | SIMULATION;
	    break;

          case 'c': 
            win->show = DATA | COMPONENTS;
	    break;

	  case 'd':
	    win->show = DIFFERENCE;
	    break;

	  case 'e':
	    win->show = DATA;
	    break; 

	  case 's':
	    win->show = SIMULATION;
	    break; 
          }                    /* End switch txt */

          DrawSpectra( win );   /* Redraw the window */
	}
       break;               /* end case KeyPress */
  
      }     /* End switch (myevent.type) */
#endif
 }



/*********************************************************************
 *                                                                   *
 *   pltwin: plot a window (FORTRAN-callable)                        *
 *                                                                   *
 *    This routine must be passed the arrays containing the data,    *
 *    the current residuals vector, and an array with the individual *
 *    component spectra                                              *
 *                                                                   *
 *********************************************************************/


void pltwin( exptl, resid, cmpts, sfac, ldcmpt, npt, nsi, wnum )
double *exptl, *resid, *cmpts, *sfac;
int *ldcmpt, *npt, *nsi, *wnum;                 
{
int i, wn;

  if (!DisplayOK) return;

  if (n_win) {
    wn = (*wnum)-1;
    if ( (wn < n_win) && (wn >= 0) ) {
      wds[wn].npts  = *npt;     /* Load locations of data/fit arrays */
      wds[wn].ncmp  = *nsi;
      wds[wn].expt = exptl;
      wds[wn].diff  = resid;
      wds[wn].scale = sfac;
      wds[wn].comp[0] = cmpts;
      for (i=1; i<wds[wn].ncmp; i++) { 
	wds[wn].comp[i] = wds[wn].comp[i-1] + *ldcmpt;
      }

      DrawSpectra( &wds[wn] );
    }
  }
}

/******************************************************************
 *                                                                *
 *  wpoll: Poll all active windows for events  (FORTRAN-callable) *
 *                                                                *
 ******************************************************************/

void wpoll()
{
  int wn, count;

  if (!DisplayOK) return;

  if (n_win) {
    for (wn = 0; wn < n_win; wn++) {
      do {
        count = XEventsQueued( wds[wn].dspy, QueuedAfterReading );
        if (count) NLSLevent( &wds[wn] );
      }
      while (count);
    }
  }
}

/***********************************************************************
 *                                                                     *
 *  shutwindows():  Close all open windows and disconnect from display *
 *                  (FORTRAN-callable)                                 * 
 *                                                                     *
 ***********************************************************************/

void shutwindows()
{
  int wn,i;

  if (!DisplayOK) return;

  if (n_win) {
    for (wn = 0; wn < n_win; wn++) {

     /* deallocate pixel array storage and delete window */
 
      if (wds[wn].pxexpt) free( wds[wn].pxexpt );
      if (wds[wn].pxdiff) free( wds[wn].pxdiff );
      if (wds[wn].pxsim) free( wds[wn].pxsim );
      for (i=0; i<wds[wn].ncmp; i++ ) 
        if (wds[wn].pxcomp[i]) free( wds[wn].pxcomp[i] );

      XFreeGC( wds[wn].dspy, wds[wn].gc );
      XDestroyWindow( wds[wn].dspy, wds[wn].wdw );
      XCloseDisplay( wds[wn].dspy );
    }
  }
}



